<?php

namespace Rbs\Payments;

use Exception;
use Bitrix\Main\EventResult;
use Rbs\Payments\Gateway;
use Bitrix\Sale\Order;
use Bitrix\Main\Config\Option;
use Bitrix\Main\Loader;
use Bitrix\Main\Localization\Loc;

Loc::loadMessages(__FILE__);

class Event
{
    const MODULE_ID = 'rbs.payment';
    const ACTION_FILE = 'rbs_payment';

    public static function OnSaleStatusOrder($order_id,$status) {
        
        try {
            $module_id = self::MODULE_ID;
            $eventResult = new EventResult(EventResult::SUCCESS);
            $RBS_Gateway = new Gateway;
            
            $order = Order::load($order_id);
            $order_status = $order->getField('STATUS_ID');
            $order_is_paid = $order->isPaid();
            $fiscal_enable = Option::get($module_id, 'RBS_ENABLE_FISCALE_OPTIONS');

            if(!$order_is_paid || !$fiscal_enable) {
                return $eventResult;
            }

            $paymentCollection = $order->getPaymentCollection();
            
            foreach ($paymentCollection as $payment) {
                if ($payment->isInner()) continue;

                $result = null;
                $service = $payment->getPaySystem();
                $payment_config = $service->getParamsBusValue($payment);
                $payment_action_file = $service->getField('ACTION_FILE');
                $PS_INVOICE_ID = $payment->getField('PS_INVOICE_ID'); // orderId

                if(
                    $PS_INVOICE_ID && 
                    $payment->isPaid() && 
                    $payment_action_file === self::ACTION_FILE && 
                    $order_status === $payment_config['RBS_PAYMENT_CLOSE_OFD_RECEIPT_STATUS']
                ) {

                    $prod_url = self::getGateProdUrl(
                        $payment_config['API_LOGIN'],
                        Option::get($module_id, 'RBS_PROD_URL_ALT_PREFIX'),
                        Option::get($module_id, 'RBS_PROD_URL'),
                        Option::get($module_id, 'RBS_PROD_URL_ALTERNATIVE_DOMAIN')
                    );

                    $RBS_Gateway->setOptions(array(
                        // module settings
                        'gate_url_prod' => Option::get($module_id, 'RBS_PROD_URL'),
                        'gate_url_test' => Option::get($module_id, 'RBS_TEST_URL'),
                        'gate_url_alternative_prod' => Option::get($module_id, 'RBS_PROD_URL_ALTERNATIVE_DOMAIN'),
                        'gate_url_alternative_test' => Option::get($module_id, 'RBS_TEST_URL_ALTERNATIVE_DOMAIN'),
                        'test_mode' => $payment_config['API_TEST_MODE'] == 'Y' ? 1 : 0,
                        'handler_logging' =>  $payment_config['HANDLER_LOGGING'] == 'Y' ? 1 : 0,

                        'ofd_enabled' => $payment_config['OFD_RECIEPT']  == 'Y' ? 1 : 0,
                        'ofd_tax' => $payment_config['OFD_TAX_SYSTEM'] == 0 ? 0 : $payment_config['RBS_OFD_TAX_SYSTEM'],
                        'ofd_enabled' => $payment_config['OFD_RECIEPT']  == 'Y' ? 1 : 0,
                        'ffd_version' => $payment_config['FFD_VERSION'],
                        'ffd_payment_object' => $payment_config['FFD_PAYMENT_OBJECT'],
                        'ffd_payment_object_delivery' => $payment_config['FFD_PAYMENT_OBJECT_DELIVER'],
                        'ffd_payment_method' => 4,
                        'ffd_payment_method_delivery' => 4,
                        'handler_two_stage' => $payment_config['HANDLER_TWO_STAGE'] == 'Y' ? 1 : 0,
                        'enable_cacert' => $payment_config['API_ENABLE_CACERT']  == 'Y' ? true : false,
                        'fes_cashboxId' => Option::get($module_id, 'RBS_ENABLE_FES_CODES') ? $payment_config['RBS_FES_CASHBOX_ID'] : "",
                        'enable_order_payer_data' => Option::get($module_id, 'RBS_ENABLE_ORDER_PAYER_DATA'),
                    ));
                    $data = array(
                        'mdOrder' => $PS_INVOICE_ID,
                        'userName' => $payment_config['API_LOGIN'],
                        'password' => $payment_config['API_PASSWORD'],
                        'amount' => $payment_config['RBS_ORDER_AMOUNT']
                    );

                    $positions = self::createOrderPositions($order, $RBS_Gateway);

                    foreach ($positions as $key => $position) {
                        $RBS_Gateway->setPosition($position);
                    }
                    $gateResponse = $RBS_Gateway->closeOfdReceipt($data);
                }

            }
        }
        catch (Exception $ex)
        {
            Logger::log($e->getMessage(), 'status_event');
        }
    }

    private static function createOrderPositions($order, $RBS_Gateway) {
        $module_id = self::MODULE_ID;
            
        $basket = $order->getBasket();
        $basket_items = $basket->getBasketItems();
        $positions = [];
        $lastIndex = 0;
        $nomenclature_code_value = Option::get($module_id, 'NOMECLATURE_CODE_VALUE');

        foreach ($basket_items as $key => $basket_item) {
            $fes_truCode = false;
            $nomenclature = false;
            if(Option::get($module_id, 'RBS_ENABLE_FES_CODES')) {
                $productId = $basket_item->getProductId(); // Получаем ID товара
                // Запрашиваем свойство "FES" у товара из инфоблока
                $productData = \CIBlockElement::GetList(
                    [],
                    ['ID' => $productId],
                    false,
                    false,
                    ['ID', 'IBLOCK_ID', 'PROPERTY_FES_TRUCODE','PROPERTY_' . $nomenclature_code_value]
                );
                if ($product = $productData->GetNext()) {
                    $fes_truCode = $product['PROPERTY_FES_TRUCODE_VALUE'] ? $product['PROPERTY_FES_TRUCODE_VALUE'] : false;
                    $nomenclature = $product['PROPERTY_'.  strtoupper($nomenclature_code_value) .'_VALUE'] ? $product['PROPERTY_'.  strtoupper($nomenclature_code_value) .'_VALUE'] : false;
                }
            }

            $lastIndex = $key + 1;
            $position = array(
                'positionId' => $key,
                'itemCode' => $basket_item->getProductId(),
                'name' => str_replace("\n", "", mb_substr($basket_item->getField('NAME'), 0, 120)),
                'itemAmount' => $basket_item->getFinalPrice(),
                'itemPrice' => $basket_item->getPrice(),
                'quantity' => array(
                    'value' => $basket_item->getQuantity(),
                    'measure' => $basket_item->getField('MEASURE_NAME') ? $basket_item->getField('MEASURE_NAME') : Loc::getMessage('RBS_FIELD_MEASURE'),
                ),
                'tax' => array(
                    'taxType' =>  $RBS_Gateway->getTaxCode( $basket_item->getField('VAT_RATE')),
                ),
                'fes_truCode' => $fes_truCode,
                'nomenclature' => $nomenclature,
            );
            
            $positions[] = $position;
        }

        if($order->getField('PRICE_DELIVERY') > 0) {
            
            Loader::includeModule('catalog');
            $deliveryInfo = \Bitrix\Sale\Delivery\Services\Manager::getById($order->getField('DELIVERY_ID'));

            $deliveryVatItem = \CCatalogVat::GetByID($deliveryInfo['VAT_ID'])->Fetch();
            $RBS_Gateway->setOptions(array(
                'delivery' => true,
            ));
            $positions[] = array(
                'positionId' => $lastIndex + 1,
                'itemCode' => 'DELIVERY_' . $order->getField('DELIVERY_ID'),
                'name' => Loc::getMessage('RBS_FIELD_DELIVERY'),
                'itemAmount' => $order->getField('PRICE_DELIVERY'),
                'itemPrice' => $order->getField('PRICE_DELIVERY'),
                'quantity' => array(
                    'value' => 1,
                    'measure' => Loc::getMessage('RBS_FIELD_MEASURE'),
                ),
                'tax' => array(
                    'taxType' => $RBS_Gateway->getTaxCodeDelivery($deliveryVatItem['RATE']),
                ),
            );
        }

        if(Option::get($module_id, 'DISCOUNT_HELPER')) {
         // DISCOUNT CALCULATE
            $RBS_Discount = new \Rbs\Payments\Discount;
            $result_order_sum = $order->getPrice() - $order->getSumPaid();
            $discount = $RBS_Discount->discoverDiscount($result_order_sum,$positions);
            if($discount > 0) { 
                $RBS_Discount->setOrderDiscount($discount);
                $recalculatedPositions = $RBS_Discount->normalizeItems($positions);
                $recalculatedAmount = $RBS_Discount->getResultAmount();
                $positions = $recalculatedPositions;
            }
        }

        
        return $positions;     
    }

    private static function getGateProdUrl($login,$prefix,$old_prod_url,$new_prod_url) {
        if(strlen($new_prod_url) > 0 && substr($login, 0, strlen($prefix)) == $prefix) {
            return $new_prod_url;
        }
        return $old_prod_url;
    }
}
